<?php
/* --------------------------------------------------------------
 OptionsResolver.php 2020-04-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2019 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Core\Configuration\Repositories\Components;

use Doctrine\DBAL\Connection;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Core\Language\TextManager;

/**
 * Class OptionsResolver
 * @package Gambio\Core\ConfigurationBak\Repository
 */
class OptionsResolver
{
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var TextManager
     */
    private $textManager;
    
    /**
     * @var int
     */
    private $languageId;
    
    
    /**
     * OptionsResolver constructor.
     *
     * @param Connection      $queryBuilder
     * @param TextManager     $textManager
     * @param UserPreferences $userPreferences
     */
    public function __construct(
        Connection $queryBuilder,
        TextManager $textManager,
        UserPreferences $userPreferences
    ) {
        $this->connection  = $queryBuilder;
        $this->textManager = $textManager;
        $this->languageId  = $userPreferences->languageId();
    }
    
    
    /**
     * Creates a new query builder.
     *
     * @return Connection
     */
    public function connection(): Connection
    {
        return $this->connection;
    }
    
    
    /**
     * Returns the requested text phrase.
     *
     * @param string $phrase
     * @param string $section
     *
     * @return string
     */
    public function getText(string $phrase, string $section): string
    {
        return $this->textManager->getPhraseText($phrase, $section, $this->languageId);
    }
    
    
    /**
     * Returns the language id of the current active language.
     *
     * @return int
     */
    public function languageId(): int
    {
        return $this->languageId;
    }
}